<?php
//"mes" o n dias preparar info reducir ruido en pronostico
function prepararDatos($datos, $agrupacion) {
    $historico = $datos["historico"];

    // 1. Ordenar por fecha ascendente
    ksort($historico);

    // 2. Convertir fechas y valores en arrays
    $fechas = array_keys($historico);
    $valores = array_values($historico);

    // 3. Agrupación por mes
    if ($agrupacion === "month") {
        $mensual = [];

        foreach ($historico as $fecha => $valor) {
            $mes = date("Y-m", strtotime($fecha));
            if (!isset($mensual[$mes])) {
                $mensual[$mes] = 0;
            }
            $mensual[$mes] += $valor;
        }

        // Eliminar el mes actual si no ha terminado
        $mes_actual = date("Y-m");
        if (isset($mensual[$mes_actual])) {
            unset($mensual[$mes_actual]);
        }

        return array_values($mensual);
    }

    // 4. Agrupación por N días (de atrás hacia adelante)
    if (is_numeric($agrupacion) && intval($agrupacion) > 0) {
        $n = intval($agrupacion);
        $agrupado = [];

        // Agrupamos de atrás hacia adelante
        $reverso = array_reverse($valores);
        $temporal = [];

        foreach ($reverso as $valor) {
            $temporal[] = $valor;
            if (count($temporal) === $n) {
                $agrupado[] = array_sum($temporal);
                $temporal = [];
            }
        }

        // Si sobran valores que no completan un grupo, se descartan

        // Revertimos de nuevo para mantener cronología
        return array_reverse($agrupado);
    }

    // Si no se reconoce la agrupación, regresa los valores originales ordenados
    return $valores;
}

//Funciones maestras
function calcular_todos_los_modelos($demanda) {
    return [
        "simple_average" => modelo_promedio_simple($demanda),
        "simple_moving_average" => modelo_media_movil_simple($demanda),
        "weighted_moving_average" => modelo_media_movil_ponderada($demanda),
        "simple_exponential_smoothing" => modelo_suavizado_exponencial_simple($demanda),
        "double_exponential_smoothing" => modelo_suavizado_exponencial_doble($demanda),
        "linear_trend" => modelo_tendencia_lineal($demanda),
        "triple_exponential_holt_winters" => modelo_holt_winters($demanda),
    ];
}
function mostrar_tabla_modelos($resultados) {
    // Encontrar el modelo con menor MAPE (ignorando null)
    $min_mape = null;
    $mejor_modelo = null;

    foreach ($resultados as $modelo => $datos) {
        if (is_numeric($datos["mape"])) {
            if ($min_mape === null || $datos["mape"] < $min_mape) {
                $min_mape = $datos["mape"];
                $mejor_modelo = $modelo;
            }
        }
    }

    // Generar tabla
    echo "<table border='1' cellpadding='6' cellspacing='0' style='border-collapse: collapse; font-family: sans-serif;'>";
    echo "<thead style='background-color:#f2f2f2;'>";
    echo "<tr>
            <th>Modelo</th>
            <!--<th>Backtest</th>-->
            <th>Forecast, next 12 rounded to whole</t h>
            <th>MAPE (%)</th>
            <th>MAE</th>
            <th>RMSE</th>
            <th>R²</th>
            <th>Parámetros</th>
          </tr>";
    echo "</thead>";
    echo "<tbody>";

    foreach ($resultados as $modelo => $datos) {
        $highlight = ($modelo == $mejor_modelo) ? " style='background-color:#d9fcd9; font-weight:bold;'" : "";

        // Convertir array de parámetros a texto legible
        $parametros = "—";
        if (!empty($datos["parametros"])) {
            $parametros = "";
            foreach ($datos["parametros"] as $k => $v) {
                if (is_array($v)) {
                    $parametros .= $k . ": [" . implode(",", $v) . "] ";
                } else {
                    $parametros .= $k . ": " . $v . " ";
                }
            }
            $parametros = trim($parametros);
        }

        echo "<tr$highlight>";
        echo "<td>".ucwords(str_replace("_", " ", $modelo))."</td>";
        echo "<!--<td>" . (
            isset($datos["backtest"]) && is_array($datos["backtest"])
                ? implode(", ", array_map(fn($v) => $v === null ? "—" : round($v, 2), $datos["backtest"]))
                : "—"
        ) . "</td>-->";
        echo "<td>".(is_array($datos["forecast"]) ? implode(", ", array_map(fn($v) => round($v, 0), $datos["forecast"])) : $datos["forecast"])."</td>";
        echo "<td>".($datos["mape"] !== null ? $datos["mape"] : "—")."</td>";
        echo "<td>".$datos["mae"]."</td>";
        echo "<td>".($datos["rmse"] !== null ? $datos["rmse"] : "—")."</td>";
        echo "<td>".($datos["r2"] !== null ? $datos["r2"] : "—")."</td>";
        echo "<td>".$parametros."</td>";
        echo "</tr>";
    }

    echo "</tbody>";
    echo "</table>";
}
function graficar_tabla_modelos($resultados, $demanda, $original, $agrupacion) {
    echo '<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>';

    $colores_forecast = [
        "simple_average" => "orange",
        "simple_moving_average" => "purple",
        "weighted_moving_average" => "brown",
        "simple_exponential_smoothing" => "yellow",
        "double_exponential_smoothing" => "blue",
        "linear_trend" => "black",
        "triple_exponential_holt_winters" => "red"
    ];
    $descripciones = [
        "simple_average" => "Best for chaotic or noisy data.",
        "simple_moving_average" => "Best for short-term smoothing.",
        "weighted_moving_average" => "Gives more weight to recent data.",
        "simple_exponential_smoothing" => "Favors recent stable data.",
        "double_exponential_smoothing" => "Captures trend without seasonality.",
        "linear_trend" => "Best for steadily trending data.",
        "triple_exponential_holt_winters" => "Captures both trend and seasonality."
    ];

    // Reconstruir fechas agrupadas
    $historico = $original["historico"];
    ksort($historico);
    $fechas_raw = array_keys($historico);

    $fechas = [];

    if ($agrupacion === "month") {
        $mensual = [];
        foreach ($fechas_raw as $fecha) {
            $mes = date("Y-m", strtotime($fecha));
            if (!isset($mensual[$mes])) {
                $mensual[$mes] = true;
            }
        }
        $mes_actual = date("Y-m");
        unset($mensual[$mes_actual]); // eliminar mes actual si existe
        $fechas = array_keys($mensual);

    } elseif (is_numeric($agrupacion) && intval($agrupacion) > 0) {
        $n = intval($agrupacion);
        $fechas_reverso = array_reverse($fechas_raw);
        $grupos = [];
        $tmp = [];

        foreach ($fechas_reverso as $fecha) {
            $tmp[] = $fecha;
            if (count($tmp) === $n) {
                $grupos[] = $tmp;
                $tmp = [];
            }
        }

        $grupos = array_reverse($grupos);
        foreach ($grupos as $g) {
            $fechas[] = end($g); // usar la última fecha del grupo como referencia
        }
    } else {
        $fechas = $fechas_raw;
    }

    // Completar fechas para forecast según agrupación
    $total_fechas = count($demanda) + 12;

    while (count($fechas) < $total_fechas) {
        $last = end($fechas);

        if ($agrupacion === "month") {
            $next = date("Y-m", strtotime($last . ' +1 month'));
        } elseif (is_numeric($agrupacion) && intval($agrupacion) > 0) {
            $n = intval($agrupacion);
            $next = date("Y-m-d", strtotime($last . " +$n days"));
        } else {
            $next = date("Y-m-d", strtotime($last . ' +1 day'));
        }

        $fechas[] = $next;
    }

    $i = 0;
    foreach ($resultados as $modelo => $datos) {
        $forecast = $datos["forecast"];
        $backtest = $datos["backtest"];

        if (!is_array($forecast)) continue;

        $forecast = array_map(fn($v) => round($v, 1), $forecast);
        $backtest = is_array($backtest) ? $backtest : [];

        $chart_id = "chart_" . $i++;

        $data_original = array_merge($demanda, array_fill(0, 12, null));
        $data_backtest = array_merge(array_fill(0, count($demanda) - count($backtest), null), $backtest, array_fill(0, 12, null));
        $data_forecast = array_merge(array_fill(0, count($demanda), null), $forecast);

        $titulo = ucwords(str_replace("_", " ", $modelo));
        if (!empty($datos["parametros"])) {
            $params = [];
            foreach ($datos["parametros"] as $k => $v) {
                if (is_array($v)) {
                    $params[] = $k . "=" . implode(",", $v);
                } else {
                    $params[] = $k . "=" . $v;
                }
            }
            $titulo .= " (" . implode(", ", $params) . ")";
        }
        if (is_numeric($datos["mape"])) {
            $titulo .= " | MAPE: " . $datos["mape"] . "%";
        }
        $titulo .= " | " .$descripciones[$modelo];

        echo "<hr style='width:50%; margin: 20px 25%;'><canvas id='$chart_id' height='400'></canvas>";
        echo "<script>
        new Chart(document.getElementById('$chart_id'), {
            type: 'line',
            data: {
                labels: " . json_encode($fechas) . ",
                datasets: [
                    {
                        label: 'Original Demand',
                        data: " . json_encode($data_original) . ",
                        borderColor: 'teal',
                        borderWidth: 1,
                        borderDash: [5,5],
                        fill: false
                    },
                    {
                        label: 'Backtest',
                        data: " . json_encode($data_backtest) . ",
                        borderColor: 'green',
                        borderWidth: 2,
                        fill: false
                    },
                    {
                        label: 'Forecast',
                        data: " . json_encode($data_forecast) . ",
                        borderColor: '" . ($colores_forecast[$modelo] ?? 'red') . "',
                        borderWidth: 2,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                plugins: {
                    tooltip: {
                        mode: 'index',
                        intersect: false
                    },
                    title: {
                        display: true,
                        text: '".$titulo."'
                    }
                },
                interaction: {
                    mode: 'nearest',
                    axis: 'x',
                    intersect: false
                },
                scales: {
                    x: {
                        title: { display: true, text: 'Date' }
                    },
                    y: {
                        title: { display: true, text: 'Value' }
                    }
                }
            }
        });
        </script>";
    }
}

//Calculo Error
function calcular_mape($reales, $predichos) {
    $total = 0;
    $n = count($reales);
    $validos = 0;

    for ($i = 0; $i < $n; $i++) {
        if ($reales[$i] != 0) {
            $total += abs(($reales[$i] - $predichos[$i]) / $reales[$i]);
            $validos++;
        }
    }

    return $validos > 0 ? round(($total / $validos) * 100, 2) : null;
}
function calcular_mae($reales, $predichos) {
    $total = 0;
    $n = count($reales);
    for ($i = 0; $i < $n; $i++) {
        $total += abs($reales[$i] - $predichos[$i]);
    }
    return round($total / $n, 2);
}
function calcular_rmse($reales, $predichos) {
    $total = 0;
    $n = count($reales);
    for ($i = 0; $i < $n; $i++) {
        $total += pow($reales[$i] - $predichos[$i], 2);
    }
    return round(sqrt($total / $n), 2);
}
function calcular_r2($reales, $predichos) {
    $media = array_sum($reales) / count($reales);
    $ss_tot = 0;
    $ss_res = 0;

    for ($i = 0; $i < count($reales); $i++) {
        $ss_tot += pow($reales[$i] - $media, 2);
        $ss_res += pow($reales[$i] - $predichos[$i], 2);
    }

    return $ss_tot == 0 ? null : round(1 - ($ss_res / $ss_tot), 2);
}

// Modelo 1: Promedio simple
function modelo_promedio_simple($valores) {
    $promedio = array_sum($valores) / count($valores);
    
    $backtest = array_fill(0, count($valores), $promedio);
    $forecast = array_fill(0, 12, round($promedio, 2)); // 12 periodos futuros

    return [
        "forecast" => $forecast,
        "backtest" => $backtest,
        "parametros" => [],
        "mape" => calcular_mape($valores, $backtest),
        "mae" => calcular_mae($valores, $backtest),
        "rmse" => null,
        "r2" => null
    ];
}
// Modelo 2: Media móvil simple (N)
function modelo_media_movil_simple($valores) {
    $mejor_mape = null;
    $mejor_resultado = [];

    for ($n = 1; $n <= min(30, count($valores) - 1); $n++) {
        $pred = [];

        // Backtest: aplicar media móvil con ventana n
        for ($i = 0; $i < count($valores); $i++) {
            if ($i >= $n) {
                $window = array_slice($valores, $i - $n, $n);
                $pred[] = array_sum($window) / $n;
            } else {
                $pred[] = null; // No hay suficientes datos para el backtest en este punto
            }
        }

        // Forecast: multistep usando los últimos n valores (incluye predichos anteriores)
        $historial = $valores;
        $forecast = [];

        for ($i = 0; $i < 12; $i++) {
            $window = array_slice($historial, -$n);
            $next = array_sum($window) / $n;
            $forecast[] = round($next, 2);
            $historial[] = $next; // Usar predicción anterior para el siguiente paso
        }

        // Evaluar error solo en valores válidos
        $valores_evaluados = array_slice($valores, $n);
        $pred_evaluados = array_slice($pred, $n);

        $mape = calcular_mape($valores_evaluados, $pred_evaluados);
        $mae = calcular_mae($valores_evaluados, $pred_evaluados);
        $score = ($mape === null) ? $mae : $mape;

        if ($mejor_mape === null || $score < $mejor_mape) {
            $mejor_mape = $score;
            $mejor_resultado = [
                "forecast" => $forecast,
                "backtest" => $pred,
                "parametros" => ["n" => $n],
                "mape" => $mape,
                "mae" => $mae,
                "rmse" => null,
                "r2" => null
            ];
        }
    }

    return $mejor_resultado;
}
// Modelo 3: Media móvil ponderada (N<6, pesos<4) - la mas pesada por combinatorias de pesos, si quieres n=12 peso<3 o truena
function modelo_media_movil_ponderada($valores) {
    if (count($valores) < 13) {
        return [
            "forecast" => "Insufficient data < 13",
            "backtest" => null,
            "parametros" => null,
            "mape" => "ERROR",
            "mae" => "ERROR",
            "rmse" => null,
            "r2" => null
        ];
    }

    $rango_pesos = [1, 2, 3, 4, 5];
    $mejor_score = null;
    $mejor_resultado = [];

    for ($n = 2; $n <= 6; $n++) {
        // Generar combinaciones de pesos para este n
        $combinaciones = [[]];
        for ($i = 0; $i < $n; $i++) {
            $tmp = [];
            foreach ($combinaciones as $combo) {
                foreach ($rango_pesos as $peso) {
                    $tmp[] = array_merge($combo, [$peso]);
                }
            }
            $combinaciones = $tmp;
        }

        foreach ($combinaciones as $pesos) {
            $suma_pesos = array_sum($pesos);
            $pred = [];

            // Backtest
            for ($i = 0; $i < count($valores); $i++) {
                if ($i >= $n) {
                    $window = array_slice($valores, $i - $n, $n);
                    $w_sum = 0;
                    for ($j = 0; $j < $n; $j++) {
                        $w_sum += $window[$j] * $pesos[$j];
                    }
                    $pred[] = $w_sum / $suma_pesos;
                } else {
                    $pred[] = null; // sin datos suficientes
                }
            }

            // Forecast dinámico a 12 pasos
            $historial = $valores;
            $forecast = [];

            for ($i = 0; $i < 12; $i++) {
                $window = array_slice($historial, -$n);
                $w_sum = 0;
                for ($j = 0; $j < $n; $j++) {
                    $w_sum += $window[$j] * $pesos[$j];
                }
                $next = $w_sum / $suma_pesos;
                $forecast[] = round($next, 2);
                $historial[] = $next;
            }

            // Errores usando solo valores evaluables
            $valores_eval = array_slice($valores, $n);
            $pred_eval = array_slice($pred, $n);

            $mape = calcular_mape($valores_eval, $pred_eval);
            $mae = calcular_mae($valores_eval, $pred_eval);
            $score = ($mape === null) ? $mae : $mape;

            if ($mejor_score === null || $score < $mejor_score) {
                $mejor_score = $score;
                $mejor_resultado = [
                    "forecast" => $forecast,
                    "backtest" => $pred,
                    "parametros" => [
                        "n" => $n,
                        "weights" => $pesos
                    ],
                    "mape" => $mape,
                    "mae" => $mae,
                    "rmse" => null,
                    "r2" => null
                ];
            }
        }
    }

    return $mejor_resultado;
}
// Modelo 4: Suavizado exponencial simple (α de 0-1)
function modelo_suavizado_exponencial_simple($valores) {
    $mejor_mape = null;
    $mejor_resultado = [];

    for ($alpha = 0.01; $alpha < 1.0; $alpha += 0.01) {
        $pred = [$valores[0]];

        // Backtest
        for ($i = 1; $i < count($valores); $i++) {
            $pred[] = $alpha * $valores[$i - 1] + (1 - $alpha) * $pred[$i - 1];
        }

        // Forecast dinámico: usar el último valor predicho como base
        $last = end($pred);
        $forecast = [];
        for ($i = 0; $i < 12; $i++) {
            $last = $alpha * $last + (1 - $alpha) * $last; // mismo valor suavizado
            $forecast[] = round($last, 2);
        }

        // Evaluar error
        $valores_eval = array_slice($valores, 1);
        $pred_eval = array_slice($pred, 1);

        $mape = calcular_mape($valores_eval, $pred_eval);
        $mae = calcular_mae($valores_eval, $pred_eval);
        $score = ($mape === null) ? $mae : $mape;

        if ($mejor_mape === null || $score < $mejor_mape) {
            $mejor_mape = $score;
            $mejor_resultado = [
                "forecast" => $forecast,
                "backtest" => $pred,
                "parametros" => ["alpha" => round($alpha, 2)],
                "mape" => $mape,
                "mae" => $mae,
                "rmse" => null,
                "r2" => null
            ];
        }
    }

    return $mejor_resultado;
}
// Modelo 5: Suavizado exponencial doble (α, β de 0-1)
function modelo_suavizado_exponencial_doble($valores) {
    $mejor_mape = null;
    $mejor_resultado = [];

    for ($alpha = 0.1; $alpha <= 1.0; $alpha += 0.01) {
        for ($beta = 0.1; $beta <= 1.0; $beta += 0.01) {
            $l = $valores[0];
            $b = $valores[1] - $valores[0];
            $pred = [$l + $b];

            for ($i = 1; $i < count($valores); $i++) {
                $last_l = $l;
                $l = $alpha * $valores[$i] + (1 - $alpha) * ($l + $b);
                $b = $beta * ($l - $last_l) + (1 - $beta) * $b;
                $pred[] = $l + $b;
            }

            // Forecast multistep
            $forecast = [];
            for ($m = 1; $m <= 12; $m++) {
                $forecast[] = round($l + $m * $b, 2);
            }

            // Errores solo a partir de i=1
            $valores_eval = array_slice($valores, 1);
            $pred_eval = array_slice($pred, 1);

            $mape = calcular_mape($valores_eval, $pred_eval);
            $mae = calcular_mae($valores_eval, $pred_eval);
            $score = ($mape === null) ? $mae : $mape;

            if ($mejor_mape === null || $score < $mejor_mape) {
                $mejor_mape = $score;
                $mejor_resultado = [
                    "forecast" => $forecast,
                    "backtest" => $pred,
                    "parametros" => [
                        "alpha" => round($alpha, 2),
                        "beta" => round($beta, 2)
                    ],
                    "mape" => $mape,
                    "mae" => $mae,
                    "rmse" => null,
                    "r2" => null
                ];
            }
        }
    }

    return $mejor_resultado;
}
// Modelo 6: Tendencia lineal
function modelo_tendencia_lineal($valores) {
    if (count($valores) < 24) {
        return [
            "forecast" => "Insufficient data < 24",
            "backtest" => null,
            "parametros" => null,
            "mape" => "ERROR",
            "mae" => "ERROR",
            "rmse" => null,
            "r2" => null
        ];
    }

    $n = count($valores);
    $x = range(0, $n - 1);

    $sum_x = array_sum($x);
    $sum_y = array_sum($valores);
    $sum_xx = array_sum(array_map(fn($xi) => $xi * $xi, $x));

    $sum_xy = 0;
    for ($i = 0; $i < $n; $i++) {
        $sum_xy += $x[$i] * $valores[$i];
    }

    // Coeficientes de la recta
    $m = ($n * $sum_xy - $sum_x * $sum_y) / ($n * $sum_xx - $sum_x * $sum_x);
    $b = ($sum_y - $m * $sum_x) / $n;

    // Backtest (predicciones sobre la serie real)
    $pred = array_map(fn($xi) => $m * $xi + $b, $x);

    // Forecast para los próximos 12 periodos
    $forecast = [];
    for ($i = $n; $i < $n + 12; $i++) {
        $forecast[] = round($m * $i + $b, 2);
    }

    return [
        "forecast" => $forecast,
        "backtest" => $pred,
        "parametros" => [], // no hay hiperparámetros
        "mape" => calcular_mape($valores, $pred),
        "mae" => calcular_mae($valores, $pred),
        "rmse" => calcular_rmse($valores, $pred),
        "r2" => calcular_r2($valores, $pred)
    ];
}
// Modelo 7: Holt-winters (suavizado exponencial triple) detecta estacionalidad
function modelo_holt_winters($valores) {
    if (count($valores) < 24) {
        return [
            "forecast" => "Insufficient data < 24",
            "backtest" => null,
            "parametros" => null,
            "mape" => "ERROR",
            "mae" => "ERROR",
            "rmse" => null,
            "r2" => null
        ];
    }

    $mejor_score = null;
    $mejor_resultado = [];

    for ($L = 6; $L <= 12; $L++) {
        if (count($valores) < 2 * $L) continue;

        for ($a = 0.1; $a <= 0.9; $a += 0.1) {
            for ($b = 0.1; $b <= 0.9; $b += 0.1) {
                for ($g = 0.1; $g <= 0.9; $g += 0.1) {

                    $level = array_sum(array_slice($valores, 0, $L)) / $L;
                    $trend = (array_sum(array_slice($valores, $L, $L)) - array_sum(array_slice($valores, 0, $L))) / ($L * $L);

                    $estacionalidad = [];
                    for ($i = 0; $i < $L; $i++) {
                        $estacionalidad[$i] = $valores[$i] - $level;
                    }

                    $predichos = [];

                    for ($t = 0; $t < count($valores); $t++) {
                        $s = $estacionalidad[$t % $L];
                        $last_level = $level;
                        $level = $a * ($valores[$t] - $s) + (1 - $a) * ($level + $trend);
                        $trend = $b * ($level - $last_level) + (1 - $b) * $trend;
                        $estacionalidad[$t % $L] = $g * ($valores[$t] - $level) + (1 - $g) * $s;

                        $predichos[] = ($level + $trend) + $estacionalidad[$t % $L];
                    }

                    // Forecast multistep
                    $forecast = [];
                    for ($m = 1; $m <= 12; $m++) {
                        $pos = (count($valores) + $m - 1) % $L;
                        $f = ($level + $m * $trend) + $estacionalidad[$pos];
                        $forecast[] = round($f, 2);
                    }

                    // Verificar sobreajuste
                    $tiene_negativos = array_filter($forecast, fn($v) => $v < 0);

                    // Errores solo comparables
                    $mape = calcular_mape($valores, $predichos);
                    $mae = calcular_mae($valores, $predichos);

                    $score = $mape !== null ? $mape : $mae;

                    if ($mejor_score === null || $score < $mejor_score) {
                        $mejor_score = $score;
                        $mejor_resultado = [
                            "forecast" => $forecast,
                            "backtest" => $predichos,
                            "parametros" => [
                                "alpha" => round($a, 1),
                                "beta" => round($b, 1),
                                "gamma" => round($g, 1),
                                "L" => $L
                            ],
                            "mape" => $tiene_negativos ? round($mape, 2)." (Warning: Overfitting)" : round($mape, 2),
                            "mae" => round($mae, 2),
                            "rmse" => null,
                            "r2" => null
                        ];
                    }
                }
            }
        }
    }

    return $mejor_resultado;
}
